/******************************************************************************/
/* Hardware : ET-BASE PIC32MX                                                 */
/* MCU      : PIC32MX250F128B                                                 */
/* Crystal  : 8MHz                                                            */
/* Run      : External Crystal(8.00MHz)+PLL                                   */
/*          : System Clock FCY = 48MHz(8MHz+PLL : 8MHz / 2 * 24 / 2)          */
/* Compiler : Microchip MPLAB XC32 v1.31                                      */
/******************************************************************************/

#ifdef __XC32
  #include <xc.h>            /* Defines special funciton registers, CP0 regs  */
#endif

#include <plib.h>           /* Include to use PIC32 peripheral libraries      */

/******************************************************************************/
/* Start of Config:PIC32MX1XX/2XX                                             */
/* define Config Bit Name  in File : "pic32mx250f128b.h"                      */
/* default: "...\Microchip\xc32\v1.31\pic32-libs\include\proc\p32mx250f128b.h"*/
/* define Config Bit Value in File : "32mx250f128b.html"                      */
/* default: "...\Microchip\xc32\v1.31\docs\config_docs\32mx250f128b.html"     */
/******************************************************************************/
// Configuration Bit settings
// Primary Osc = w / PLL (XT+,HS+,EC+PLL)
//             = POSCMOD : HS
//             = FNOSC   : PRIPLL
// SYSCLK      = 8MHz Crystal / FPLLIDIV(2) * FPLLMUL(24) / FPLLODIV(2)
//             = 8.0 MHz      / 2           * 24          / 2
//             = 48.0 MHz
// PBCLK       = SYSCLK(48)   / FPBDIV(1)
//             = 48.0MHz / 1
//             = 48.0 MHz
// WDT OFF

// DEVCFG[0]
#pragma config DEBUG = ON            // Debugger is Enabled
#pragma config JTAGEN = OFF          // JTAG Disabled
#pragma config ICESEL = ICS_PGx1     // ICD Communicate on PGEC1/PGED1
#pragma config PWP = OFF             // Program Flash Write Protect = Disable
#pragma config BWP = OFF             // Boot Flash Write Protect = Disabled
#pragma config CP = OFF              // Code Protect = Disabled

// DEVCFG[1]
#pragma config FNOSC = PRIPLL        // Primary Osc w/PLL (XT+,HS+,EC+PLL)
#pragma config POSCMOD = HS          // Primary OSC = HS OSC Mode
#pragma config OSCIOFNC = ON         // CLKO Output on the OSCO = Enabled

#pragma config FSOSCEN = OFF         // Secondary Oscillator = Disabled
#pragma config IESO = OFF            // Internal/External Switch Over=Disabled
#pragma config FPBDIV = DIV_1        // PBCLK = SYSCLK / 1
#pragma config FCKSM = CSDCMD        // Clock Switch Disable, FSCM Disabled
#pragma config WDTPS = PS1048576     // Watchdog Timer Postscaler = 1:1048576
#pragma config WINDIS = OFF          // Watchdog Timer is in Non-Window Mode
#pragma config FWDTEN = OFF          // WDT Disabled (SWDTEN Bit Controls)
#pragma config FWDTWINSZ = WINSZ_75  // Watchdog Timer Window Size is 75%

// DEVCFG[2]
#pragma config FPLLIDIV = DIV_2      // PLL Input Divider = Div 2
#pragma config FPLLMUL = MUL_24      // Fin = PLL x 24
#pragma config UPLLIDIV = DIV_2      // Input USB PLL = Input Clock / 2
#pragma config UPLLEN = OFF          // USB PLL = Disabled and Bypassed
#pragma config FPLLODIV = DIV_2      // Output USB PLL = Output Clock / 2

// DEVCFG[3]
//#pragma config USERID = OFF          // USB USID = Controlled by Port Function
#pragma config PMDL1WAY = OFF        // Peripheral Module = Allow multiple reconfigurations
#pragma config IOL1WAY = OFF         // Peripheral Pin Select = Allow multiple reconfigurations
#pragma config FUSBIDIO = OFF        // USB USID = Controlled by Port Function
#pragma config FVBUSONIO = OFF       // USB VBUS ON = Controlled by Port Function

#define SYS_FREQ (48000000L)         // Run 48.0 MHz
#define LONG_DELAY  (SYS_FREQ/100)

#define	GetSystemClock() 		(48000000ul)
#define	GetPeripheralClock()		(GetSystemClock()/(1 << OSCCONbits.PBDIV))
#define	GetInstructionClock()		(GetSystemClock())

/******************************************************************************/
/* End of Config:PIC32MX1XX/2XX                                               */
/******************************************************************************/
#define OUTPUT         0
#define INPUT          1
#define LED_ON         0
#define LED_OFF        1

/******************************************************************************/
/* Global Variable Declaration                                                */
/******************************************************************************/
void delay(unsigned long delay_count);

void SendDataBufferUART1(const char *buffer, UINT32 size);
UINT32 GetDataBufferUART1(char *buffer, UINT32 max_size);
UINT32 GetMenuChoiceUART1(void);

void SendDataBufferUART2(const char *buffer, UINT32 size);
UINT32 GetDataBufferUART2(char *buffer, UINT32 max_size);
UINT32 GetMenuChoiceUART2(void);

const char mainMenuUART1[] =
{
  "\r\n\r\n"\
  "=======================================\r\n"\
  "ET-BASE PIC32 : PIC32MX250F128B UART[1]\r\n"\
  "Run External Crystal(8MHz)+PLL = 48MHz\r\n"\
  "Used PIC32 Peripheral Library Demo\r\n"\
  "=======================================\r\n"\
  "UART1>"
};

// Constant Data
const char mainMenuUART2[] =
{
  "\r\n\r\n"\
  "=======================================\r\n"\
  "ET-BASE PIC32 : PIC32MX250F128B UART[2]\r\n"\
  "Run External Crystal(8MHz)+PLL = 48MHz\r\n"\
  "Used PIC32 Peripheral Library Demo \r\n"\
  "======================================= \r\n"\
  "UART2>"
};

/******************************************************************************/
/* Main Program                                                               */
/******************************************************************************/

void main(void)
{
  UINT32  rx_data;
  unsigned long i;

  //============================================================================
  // Configure the device for maximum performance, but do not change the PBDIV clock divisor.
  // Given the options, this function will change the program Flash wait states,
  // RAM wait state and enable prefetch cache, but will not change the PBDIV.
  // The PBDIV value is already set via the pragma FPBDIV option above.
  SYSTEMConfig(SYS_FREQ, SYS_CFG_WAIT_STATES | SYS_CFG_PCACHE); 
  //============================================================================

  //Mapping RA4=RX1,RA0=TX1
  PPSInput(3,U1RX,RPA4);                                                        // Assign RPA4 as input pin for U1RX
  PPSOutput(1,RPA0,U1TX);                                                       // Set RPA0 pin as output for U1TX

  //Mapping RB8=RX2,RB9=TX2
  PPSInput(2,U2RX,RPB8);                                                        // Assign RPB8 as input pin for U2RX
  PPSOutput(4,RPB9,U2TX);                                                       // Set RPB9 pin as output for U2TX

  //Initial UART1
  UARTConfigure(UART1, UART_ENABLE_PINS_TX_RX_ONLY);
  UARTSetFifoMode(UART1, UART_INTERRUPT_ON_TX_NOT_FULL | UART_INTERRUPT_ON_RX_NOT_EMPTY);
  UARTSetLineControl(UART1, UART_DATA_SIZE_8_BITS | UART_PARITY_NONE | UART_STOP_BITS_1);
  UARTSetDataRate(UART1, GetPeripheralClock(), 9600);
  UARTEnable(UART1, UART_ENABLE_FLAGS(UART_PERIPHERAL | UART_RX | UART_TX));

  //Initial UART2
  UARTConfigure(UART2, UART_ENABLE_PINS_TX_RX_ONLY);
  UARTSetFifoMode(UART2, UART_INTERRUPT_ON_TX_NOT_FULL | UART_INTERRUPT_ON_RX_NOT_EMPTY);
  UARTSetLineControl(UART2, UART_DATA_SIZE_8_BITS | UART_PARITY_NONE | UART_STOP_BITS_1);
  UARTSetDataRate(UART2, GetPeripheralClock(), 9600);
  UARTEnable(UART2, UART_ENABLE_FLAGS(UART_PERIPHERAL | UART_RX | UART_TX));

  //Display Menu On UART
  SendDataBufferUART1(mainMenuUART1, sizeof(mainMenuUART1));
  SendDataBufferUART2(mainMenuUART2, sizeof(mainMenuUART2));
  
  PORTSetPinsDigitalOut(IOPORT_B, BIT_14);
  PORTSetPinsDigitalOut(IOPORT_B, BIT_15);  
  PORTSetBits(IOPORT_B, BIT_14);                                                // ON LED1
  PORTClearBits(IOPORT_B, BIT_15);                                              // OFF LED2
  
  while(1)
  {
    for(i=0; i<LONG_DELAY; i++)
    {
      //UART1
      if(UARTReceivedDataIsAvailable(UART1))					// If RX Receive Complete
      {
        rx_data = UARTGetDataByte(UART1);				        // Get UART1 Data Receive
        if(rx_data == 0x0D)							// If Receive Enter
        {
          SendDataBufferUART1("\r\nUART1>",9);
        }
        else
        {
          while(!UARTTransmitterIsReady(UART1));
          UARTSendDataByte(UART1, rx_data);
        }
        while(!UARTTransmissionHasCompleted(UART1));
      }

      //UART2
      if(UARTReceivedDataIsAvailable(UART2))					// If RX Receive Complete
      {
        rx_data = UARTGetDataByte(UART2);				        // Get UART1 Data Receive
        if(rx_data == 0x0D)							// If Receive Enter
        {
          SendDataBufferUART2("\r\nUART2>",9);
        }
        else
        {
          while(!UARTTransmitterIsReady(UART2));
          UARTSendDataByte(UART2, rx_data);
        }
        while(!UARTTransmissionHasCompleted(UART2));
      }
    }

    //Blink LED
    mPORTBToggleBits(BIT_14);                                                   // Blink LED1
    mPORTBToggleBits(BIT_15);                                                   // Blink LED2
  }
  //============================================================================
}

/**************************************/
/* Polling Loop Delay 1...delay_count */
/**************************************/
void delay(unsigned long delay_count)
{
  unsigned long i;
  for(i=0; i<delay_count; i++);
}

// *****************************************************************************
// void UARTTxBuffer(char *buffer, UINT32 size)
// *****************************************************************************
void SendDataBufferUART1(const char *buffer, UINT32 size)
{
  while(size-1)
  {
    while(!UARTTransmitterIsReady(UART1));
    UARTSendDataByte(UART1, *buffer);
    buffer++;
    size--;
  }
  while(!UARTTransmissionHasCompleted(UART1));
}

// *****************************************************************************
// UINT32 GetDataBuffer(char *buffer, UINT32 max_size)
// *****************************************************************************
UINT32 GetDataBufferUART1(char *buffer, UINT32 max_size)
{
  UINT32 num_char;
  num_char = 0;
  while(num_char < max_size)
  {
    UINT8 character;
    while(!UARTReceivedDataIsAvailable(UART1));
    character = UARTGetDataByte(UART1);

    if(character == '\r')
    break;

    *buffer = character;

    buffer++;
    num_char++;
  }
  return num_char;
}

// *****************************************************************************
// UINT32 GetMenuChoice(void)
// *****************************************************************************
UINT32 GetMenuChoiceUART1(void)
{
  UINT8  menu_item;

  while(!UARTReceivedDataIsAvailable(UART1));
  menu_item = UARTGetDataByte(UART1);
  menu_item -= '0';
  return (UINT32)menu_item;
}

// *****************************************************************************
// void UARTTxBuffer(char *buffer, UINT32 size)
// *****************************************************************************
void SendDataBufferUART2(const char *buffer, UINT32 size)
{
  while(size-1)
  {
    while(!UARTTransmitterIsReady(UART2));
    UARTSendDataByte(UART2, *buffer);
    buffer++;
    size--;
  }
  while(!UARTTransmissionHasCompleted(UART2));
}

// *****************************************************************************
// UINT32 GetDataBuffer(char *buffer, UINT32 max_size)
// *****************************************************************************
UINT32 GetDataBufferUART2(char *buffer, UINT32 max_size)
{
  UINT32 num_char;

  num_char = 0;
  while(num_char < max_size)
  {
    UINT8 character;
    while(!UARTReceivedDataIsAvailable(UART2));
    character = UARTGetDataByte(UART2);

    if(character == '\r')
    break;

    *buffer = character;

    buffer++;
    num_char++;
  }
  return num_char;
}

// *****************************************************************************
// UINT32 GetMenuChoice(void)
// *****************************************************************************
UINT32 GetMenuChoiceUART2(void)
{
  UINT8  menu_item;

  while(!UARTReceivedDataIsAvailable(UART2));
  menu_item = UARTGetDataByte(UART2);
  menu_item -= '0';
  return (UINT32)menu_item;
}
